#!/usr/bin/env python3
"""
Script to delete all bets from the database
"""

import sys
from pathlib import Path

# Add the project root to Python path
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

from mbetterclient.database.models import BetModel, BetDetailModel
from mbetterclient.database.manager import DatabaseManager
from mbetterclient.config.settings import get_user_data_dir

def delete_all_bets():
    """Delete all bets and their details from the database"""

    # Use the default database path
    db_path = get_user_data_dir() / "mbetterclient.db"
    db_manager = DatabaseManager(str(db_path))

    if not db_manager.initialize():
        print("Failed to initialize database")
        return False

    session = db_manager.get_session()
    try:
        # Get count of bets before deletion
        total_bets = session.query(BetModel).count()
        total_details = session.query(BetDetailModel).count()

        if total_bets == 0:
            print("No bets found in the database")
            return True

        print(f"Found {total_bets} bets with {total_details} total bet details")

        # Delete all bet details first (due to foreign key constraints)
        deleted_details = session.query(BetDetailModel).delete()
        print(f"Deleted {deleted_details} bet details")

        # Delete all bets
        deleted_bets = session.query(BetModel).delete()
        print(f"Deleted {deleted_bets} bets")

        # Commit the transaction
        session.commit()

        print("✅ Successfully deleted all bets from the database")
        return True

    except Exception as e:
        print(f"Error deleting bets: {e}")
        session.rollback()
        return False
    finally:
        session.close()
        db_manager.close()

if __name__ == "__main__":
    print("Deleting all bets from the database...")
    success = delete_all_bets()
    if success:
        print("\n🗑️ All bets have been deleted successfully!")
    else:
        print("\n❌ Failed to delete bets.")
        sys.exit(1)