#!/usr/bin/env python3
"""
Create an icon for MBetter Discovery application
Generates a .ico file for Windows PyInstaller builds
"""

try:
    from PIL import Image, ImageDraw, ImageFont
    import os
except ImportError:
    print("Error: PIL (Pillow) is required but not installed.")
    print("Please install it with: pip install Pillow")
    exit(1)

def create_discovery_icon():
    """Create a discovery-themed icon"""
    # Icon sizes for .ico file (Windows standard)
    sizes = [16, 24, 32, 48, 64, 128, 256]
    images = []
    
    for size in sizes:
        # Create image with transparent background
        img = Image.new('RGBA', (size, size), (0, 0, 0, 0))
        draw = ImageDraw.Draw(img)
        
        # Colors - network/discovery theme
        bg_color = (41, 128, 185, 255)  # Blue background
        accent_color = (52, 152, 219, 255)  # Lighter blue
        white = (255, 255, 255, 255)
        
        # Draw background circle
        margin = max(1, size // 16)
        circle_size = size - (margin * 2)
        draw.ellipse([margin, margin, margin + circle_size, margin + circle_size], 
                    fill=bg_color, outline=accent_color, width=max(1, size // 32))
        
        # Draw network/radar pattern (concentric circles)
        center = size // 2
        if size >= 32:
            # Multiple radar circles for larger icons
            for i in range(3):
                radius = (size // 6) + (i * size // 12)
                if radius < center - margin:
                    draw.ellipse([center - radius, center - radius, 
                                center + radius, center + radius], 
                                outline=white, width=max(1, size // 64))
        elif size >= 16:
            # Single circle for smaller icons
            radius = size // 4
            draw.ellipse([center - radius, center - radius, 
                        center + radius, center + radius], 
                        outline=white, width=1)
        
        # Draw center dot (discovery point)
        dot_size = max(2, size // 16)
        draw.ellipse([center - dot_size, center - dot_size, 
                     center + dot_size, center + dot_size], 
                     fill=white)
        
        # Draw signal waves (for larger icons)
        if size >= 48:
            wave_length = size // 8
            wave_width = max(1, size // 64)
            
            # Top-right wave
            start_x = center + size // 8
            start_y = center - size // 8
            draw.arc([start_x - wave_length, start_y - wave_length,
                     start_x + wave_length, start_y + wave_length], 
                     225, 315, fill=white, width=wave_width)
            
            # Bottom-right wave  
            start_x = center + size // 8
            start_y = center + size // 8
            draw.arc([start_x - wave_length, start_y - wave_length,
                     start_x + wave_length, start_y + wave_length], 
                     135, 225, fill=white, width=wave_width)
        
        images.append(img)
    
    return images

def save_icon(images, filename):
    """Save images as .ico file"""
    try:
        # Save as .ico file with multiple sizes
        images[0].save(filename, format='ICO', 
                      sizes=[(img.width, img.height) for img in images])
        print(f"✓ Icon saved as {filename}")
        
        # Also save largest as PNG for preview
        png_filename = filename.replace('.ico', '_preview.png')
        images[-1].save(png_filename, format='PNG')
        print(f"✓ Preview saved as {png_filename}")
        
        return True
    except Exception as e:
        print(f"✗ Failed to save icon: {e}")
        return False

def main():
    """Main function"""
    print("Creating MBetter Discovery icon...")
    
    try:
        # Create icon images
        images = create_discovery_icon()
        
        # Save as .ico file
        icon_filename = "mbetter_discovery_icon.ico"
        if save_icon(images, icon_filename):
            print(f"\n✓ Icon creation successful!")
            print(f"  Icon file: {icon_filename}")
            print(f"  Sizes included: {[img.size for img in images]}")
            
            # Check file size
            if os.path.exists(icon_filename):
                file_size = os.path.getsize(icon_filename)
                print(f"  File size: {file_size} bytes")
            
            return True
        else:
            return False
            
    except Exception as e:
        print(f"✗ Icon creation failed: {e}")
        return False

if __name__ == "__main__":
    success = main()
    exit(0 if success else 1)