# Reports Sync 500 Error Debugging Guide

## Current Status

✅ **Fixed**: Client now sends actual report data (no more empty JSON `{}`)
❌ **New Issue**: Server returns 500 Internal Server Error

```
2026-02-01 17:49:55 - mbetterclient.api_client.client - ERROR - API request failed: reports_sync - 
HTTPConnectionPool(host='38.242.220.147', port=5000): Max retries exceeded with url: /api/reports/sync 
(Caused by ResponseError('too many 500 error responses'))
```

## What This Means

The client is successfully collecting and sending report data to the server, but the server is encountering an internal error while processing the request. This is a **server-side issue**, not a client-side issue.

## Enhanced Logging Added

I've added detailed logging to help diagnose the issue. When the next sync occurs, you'll see:

### Client-Side Logs (Before Request):
```
DEBUG - Collecting report data for reports_sync endpoint
INFO - Collected report data: X bets, Y stats
DEBUG - Report data structure: ['sync_id', 'client_id', 'sync_timestamp', 'date_range', 'start_date', 'end_date', 'bets', 'extraction_stats', 'summary']
DEBUG - Report data keys: {"sync_id": "str", "client_id": "str", "sync_timestamp": "str", "date_range": "str", "start_date": "str", "end_date": "str", "bets": "list", "extraction_stats": "list", "summary": "dict"}
DEBUG - Sample bet data: {"uuid": "...", "fixture_id": "...", ...}
DEBUG - Sample extraction stat: {"match_id": 1, "fixture_id": "...", ...}
INFO - Reports sync payload size: XXXX bytes (X.XX KB)
```

### Client-Side Logs (After 500 Error):
```
ERROR - Reports sync server error - Status: 500
ERROR - Server response: [Server's error message]
ERROR - Request URL: https://38.242.220.147:5000/api/reports/sync
ERROR - Request method: POST
ERROR - Request headers: {'Content-Type': 'application/json', 'Authorization': 'Bearer ...', ...}
ERROR - Request JSON keys: ['sync_id', 'client_id', 'sync_timestamp', 'date_range', 'start_date', 'end_date', 'bets', 'extraction_stats', 'summary']
ERROR - Request JSON size: XXXX bytes
```

## Common Causes of 500 Error

### 1. **Database Schema Mismatch**
Server's database schema doesn't match the data structure being sent.

**Check**: Compare client's data structure with server's expected schema.

**Client Data Structure**:
```json
{
  "sync_id": "sync_20260201_120000_abc123",
  "client_id": "client_rustdesk_123",
  "sync_timestamp": "2026-02-01T12:00:00.123456",
  "date_range": "today",
  "start_date": "2026-02-01T00:00:00",
  "end_date": "2026-02-01T23:59:59",
  "bets": [
    {
      "uuid": "bet_uuid_here",
      "fixture_id": "fixture_id_here",
      "bet_datetime": "2026-02-01T10:00:00",
      "paid": true,
      "paid_out": false,
      "total_amount": 100.0,
      "bet_count": 2,
      "details": [
        {
          "match_id": 1,
          "match_number": "match_001",
          "outcome": "WIN1",
          "amount": 50.0,
          "win_amount": 0.0,
          "result": "lost"
        }
      ]
    }
  ],
  "extraction_stats": [
    {
      "match_id": 1,
      "fixture_id": "fixture_id_here",
      "match_datetime": "2026-02-01T10:00:00",
      "total_bets": 10,
      "total_amount_collected": 1000.0,
      "total_redistributed": 950.0,
      "actual_result": "WIN1",
      "extraction_result": "WIN1",
      "cap_applied": false,
      "cap_percentage": null,
      "under_bets": 5,
      "under_amount": 500.0,
      "over_bets": 5,
      "over_amount": 500.0,
      "result_breakdown": {}
    }
  ],
  "summary": {
    "total_payin": 100.0,
    "total_payout": 950.0,
    "net_profit": -850.0,
    "total_bets": 2,
    "total_matches": 1
  }
}
```

### 2. **Missing Required Fields**
Server expects certain fields that are not present in the data.

**Check**: Review server's API specification and ensure all required fields are included.

### 3. **Data Type Mismatch**
Server expects different data types than what's being sent (e.g., string instead of integer).

**Check**: Verify data types match server's expectations.

### 4. **Foreign Key Constraints**
Server's database has foreign key constraints that are being violated.

**Check**: Ensure all referenced IDs (match_id, fixture_id, etc.) exist in server's database.

### 5. **Null/Empty Values**
Server doesn't handle null or empty values properly.

**Check**: Review server logs for null value errors.

### 6. **Payload Size Limit**
Server has a maximum payload size limit that's being exceeded.

**Check**: Client logs now show payload size. Verify it's within server's limits.

### 7. **Authentication/Authorization Issues**
Server has issues with the Bearer token or user permissions.

**Check**: Verify token is valid and user has permission to sync reports.

### 8. **Server-Side Code Bug**
Server has a bug in the reports sync endpoint.

**Check**: Review server's error logs for stack traces.

## Debugging Steps

### Step 1: Check Client Logs
Look for the enhanced logging output:
```bash
# View client logs
tail -f /path/to/client.log | grep -A 20 "reports_sync"
```

**What to look for**:
- Payload size (is it too large?)
- Data structure (are all fields present?)
- Sample data (is the data format correct?)

### Step 2: Check Server Logs
Server should have detailed error logs showing the exact cause of the 500 error.

```bash
# View server logs
tail -f /path/to/server.log | grep -A 30 "reports/sync"
```

**What to look for**:
- Stack traces
- Database errors
- Validation errors
- Missing field errors

### Step 3: Compare Data Structures
Compare the client's data structure with the server's expected schema.

**Client Schema**: See [`REPORTS_SYNC_API_SPECIFICATION.txt`](REPORTS_SYNC_API_SPECIFICATION.txt)
**Server Schema**: Check server's API documentation or code

### Step 4: Test with Minimal Data
Try sending minimal data to isolate the issue:

```python
# Test with empty arrays
minimal_data = {
    "sync_id": "test_sync_001",
    "client_id": "test_client",
    "sync_timestamp": datetime.utcnow().isoformat(),
    "date_range": "today",
    "start_date": "2026-02-01T00:00:00",
    "end_date": "2026-02-01T23:59:59",
    "bets": [],
    "extraction_stats": [],
    "summary": {
        "total_payin": 0.0,
        "total_payout": 0.0,
        "net_profit": 0.0,
        "total_bets": 0,
        "total_matches": 0
    }
}
```

If this works, the issue is with the actual data content, not the structure.

### Step 5: Test with Single Item
Try sending just one bet and one stat:

```python
minimal_data = {
    "sync_id": "test_sync_002",
    "client_id": "test_client",
    "sync_timestamp": datetime.utcnow().isoformat(),
    "date_range": "today",
    "start_date": "2026-02-01T00:00:00",
    "end_date": "2026-02-01T23:59:59",
    "bets": [
        {
            "uuid": "test_bet_001",
            "fixture_id": "test_fixture",
            "bet_datetime": "2026-02-01T10:00:00",
            "paid": true,
            "paid_out": false,
            "total_amount": 100.0,
            "bet_count": 1,
            "details": [
                {
                    "match_id": 1,
                    "match_number": "test_match",
                    "outcome": "WIN1",
                    "amount": 100.0,
                    "win_amount": 0.0,
                    "result": "lost"
                }
            ]
        }
    ],
    "extraction_stats": [
        {
            "match_id": 1,
            "fixture_id": "test_fixture",
            "match_datetime": "2026-02-01T10:00:00",
            "total_bets": 1,
            "total_amount_collected": 100.0,
            "total_redistributed": 0.0,
            "actual_result": "WIN1",
            "extraction_result": "WIN1",
            "cap_applied": false,
            "cap_percentage": null,
            "under_bets": 0,
            "under_amount": 0.0,
            "over_bets": 1,
            "over_amount": 100.0,
            "result_breakdown": {}
        }
    ],
    "summary": {
        "total_payin": 100.0,
        "total_payout": 0.0,
        "net_profit": 100.0,
        "total_bets": 1,
        "total_matches": 1
    }
}
```

### Step 6: Enable Server Debug Mode
If possible, enable debug mode on the server to get detailed error information.

## Quick Fixes to Try

### Fix 1: Check Server's Database Schema
Ensure server's database has all required tables and columns:

```sql
-- Check if tables exist
SELECT name FROM sqlite_master WHERE type='table' AND name IN ('bets', 'bet_details', 'extraction_stats');

-- Check table schemas
PRAGMA table_info(bets);
PRAGMA table_info(bet_details);
PRAGMA table_info(extraction_stats);
```

### Fix 2: Add Error Handling on Server
Ensure server's reports sync endpoint has proper error handling:

```python
@app.post("/api/reports/sync")
async def reports_sync(request: Request):
    try:
        data = await request.json()
        # Process data...
        return {"success": True, "synced_count": len(data.get('bets', []))}
    except Exception as e:
        logger.error(f"Reports sync error: {e}", exc_info=True)
        return {"success": False, "error": str(e)}, 500
```

### Fix 3: Validate Data Before Processing
Add validation on server side to catch issues early:

```python
# Validate required fields
required_fields = ['sync_id', 'client_id', 'bets', 'extraction_stats']
for field in required_fields:
    if field not in data:
        return {"success": False, "error": f"Missing required field: {field}"}, 400

# Validate data types
if not isinstance(data['bets'], list):
    return {"success": False, "error": "bets must be a list"}, 400

if not isinstance(data['extraction_stats'], list):
    return {"success": False, "error": "extraction_stats must be a list"}, 400
```

## Next Steps

1. **Wait for next sync** to see enhanced logging
2. **Check client logs** for payload details
3. **Check server logs** for error details
4. **Share logs** to identify the exact cause
5. **Apply fix** based on identified issue

## Information Needed to Diagnose

To help diagnose the 500 error, please provide:

1. **Client logs** showing the request details (after next sync)
2. **Server logs** showing the error details
3. **Server's database schema** (tables and columns)
4. **Server's API endpoint code** (if available)

## Temporary Workaround

While debugging, you can disable the reports sync endpoint to prevent repeated 500 errors:

```python
# In client configuration
"reports_sync": {
    "enabled": False,  # Temporarily disable
    ...
}
```

Or remove the API token to disable all authenticated endpoints.

## Summary

✅ **Client-side fix complete**: Client now sends actual report data
❌ **Server-side issue**: Server returns 500 Internal Server Error
🔍 **Next step**: Check server logs to identify root cause

The enhanced logging will provide the information needed to diagnose and fix the server-side issue.