#!/usr/bin/env python3
"""
Simple test script to verify the timer functionality
"""

import sys
import time
import threading
from pathlib import Path

# Add the project root to Python path
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

from mbetterclient.config.settings import AppSettings
from mbetterclient.core.message_bus import MessageBus, MessageType

def test_timer():
    """Test the timer functionality"""
    print("Testing timer functionality...")

    # Create settings with timer enabled
    settings = AppSettings()
    settings.timer.enabled = True
    settings.timer.delay_minutes = 0.1  # 6 seconds for testing

    # Create message bus
    message_bus = MessageBus()

    # Variables to track messages
    start_game_received = [False]
    start_game_delayed_received = [False]

    def message_handler(message):
        if message.type == MessageType.START_GAME:
            print(f"✓ START_GAME message received from {message.sender}")
            start_game_received[0] = True

            # Simulate the application behavior - start timer when START_GAME is received
            if settings.timer.enabled:
                delay_seconds = settings.timer.delay_minutes * 60
                print(f"Starting timer for {delay_seconds} seconds...")

                def timer_callback():
                    print("Timer expired, sending START_GAME_DELAYED message")
                    from mbetterclient.core.message_bus import MessageBuilder
                    delayed_message = MessageBuilder.start_game_delayed(
                        sender="timer",
                        delay_minutes=settings.timer.delay_minutes
                    )
                    message_bus.publish(delayed_message, broadcast=True)

                timer = threading.Timer(delay_seconds, timer_callback)
                timer.daemon = True
                timer.start()

        elif message.type == MessageType.START_GAME_DELAYED:
            print(f"✓ START_GAME_DELAYED message received from {message.sender}")
            start_game_delayed_received[0] = True

    # Register handler for both message types
    message_bus.subscribe_global(MessageType.START_GAME, message_handler)
    message_bus.subscribe_global(MessageType.START_GAME_DELAYED, message_handler)

    # Send START_GAME message to trigger the timer
    print("Sending START_GAME message to trigger timer...")
    from mbetterclient.core.message_bus import MessageBuilder
    start_game_message = MessageBuilder.start_game(sender="test_trigger")
    message_bus.publish(start_game_message, broadcast=True)

    # Wait for the delayed message
    delay_seconds = settings.timer.delay_minutes * 60
    print(f"Waiting for timer to expire ({delay_seconds} seconds)...")
    time.sleep(delay_seconds + 1)

    if start_game_received[0] and start_game_delayed_received[0]:
        print("✓ Test PASSED: Both START_GAME and START_GAME_DELAYED messages were received")
        return True
    else:
        print(f"✗ Test FAILED: START_GAME received: {start_game_received[0]}, START_GAME_DELAYED received: {start_game_delayed_received[0]}")
        return False

if __name__ == "__main__":
    success = test_timer()
    sys.exit(0 if success else 1)