from __future__ import annotations

#import cv2
#import acapture

import argparse
import pyvirtualcam
from pyvirtualcam import PixelFormat
try:
    import cupy as np
    if not np.cuda.is_available():
        print("No CUDA detected, using NumPy")
        import numpy as np
    else:
        print("CUDA detected, using CuPy for NumPY")
except:
    print("No CUDA detected, using NumPy")
    import numpy as np
import time

import asyncio
import logging
import os
from asyncio import StreamReader

from pyrtmp import StreamClosedException
from pyrtmp.flv import FLVMediaType, FLVWriter
from pyrtmp.rtmp import RTMPProtocol, SimpleRTMPController, SimpleRTMPServer
from pyrtmp.session_manager import SessionManager

logging.basicConfig(level=logging.DEBUG)
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

parser = argparse.ArgumentParser()
parser.add_argument("--device", help="VirtualCam path (optional)")
parser.add_argument("--port", help="Port to listen for RTMP", default=1935, type=int)
args = parser.parse_args()


class RTMP2SocketController(SimpleRTMPController):
    def __init__(self, output_directory: str):
        self.output_directory = output_directory
        super().__init__()

    async def on_ns_publish(self, session, message) -> None:
        publishing_name = message.publishing_name
        prefix = os.path.join(self.output_directory, f"{publishing_name}")
        session.state = RemoteProcessFLVWriter()
        logger.debug(f"output to {prefix}.flv")
        await session.state.initialize(
            #command=f"ffmpeg -y -i pipe:0 -c:v copy -c:a copy -f flv {prefix}.flv",
            command=f"ffmpeg -hide_banner -y -i pipe:0 -pix_fmt bgr24 -vcodec rawvideo -an -sn -f image2pipe -",
            stdout_log=f"{prefix}.stdout.log",
            stderr_log=f"{prefix}.stderr.log",
        )
        session.state.write_header()
        await super().on_ns_publish(session, message)

    async def on_metadata(self, session, message) -> None:
        session.state.write(0, message.to_raw_meta(), FLVMediaType.OBJECT)
        await super().on_metadata(session, message)

    async def on_video_message(self, session, message) -> None:
        session.state.write(message.timestamp, message.payload, FLVMediaType.VIDEO)
        await super().on_video_message(session, message)

    async def on_audio_message(self, session, message) -> None:
        session.state.write(message.timestamp, message.payload, FLVMediaType.AUDIO)
        await super().on_audio_message(session, message)

    async def on_stream_closed(self, session: SessionManager, exception: StreamClosedException) -> None:
        await session.state.close()
        await super().on_stream_closed(session, exception)



class RemoteProcessFLVWriter:
    def __init__(self):
        self.proc = None
        self.stdout = None
        self.stderr = None
        self.writer = FLVWriter()

    async def initialize(self, command: str, stdout_log: str, stderr_log: str):
        self.proc = await asyncio.create_subprocess_shell(
            command,
            stdin=asyncio.subprocess.PIPE,
            stdout=asyncio.subprocess.PIPE,
            stderr=asyncio.subprocess.PIPE,
        )
        self.stdout = asyncio.create_task(self._read_to_vcam(self.proc.stdout))
        #self.stderr = asyncio.create_task(self._read_to_file(stderr_log, self.proc.stderr))
        #self.stdout = asyncio.create_task(self._read_to_vcam(self.proc))

    async def _read_to_vcam(self, stream: StreamReader):
        with pyvirtualcam.Camera(1920, 1080, 30, fmt=PixelFormat.BGR, device=args.device, print_fps=30) as cam:
            frame=b''
            frames=0
            last=time.time()
            while not stream.at_eof():
                #pkt = await stream.read(1920*1080*3)
                #print(len(pkt))
                pkt = await stream.readexactly(1920*1080*3)
                #pkt, stderr = await stream.communicate()
                if pkt:
                    frame = frame + pkt
                    #print(len(frame))
                    if len(frame) >= 1920*1080*3:
                        if time.time()-last >= 1:
                            last=time.time()
                            print(frames, 'fps')
                            frames=0
                        #cam.send(np.fromstring(frame[:1920*1080*3], dtype='uint8').reshape((1080,1920,3,)))
                        #cam.send(np.frombuffer(frame, dtype='uint8').reshape((1080, 1920, 3,)))
                        #cam.send(rframe)
                        #cam.sleep_until_next_frame()

                        frames=frames+1
                        frame=frame[1920*1080*3:]
                        time.sleep(.01)
          



    async def _read_to_file(self, filename: str, stream: StreamReader):
        fp = open(filename, "w")
        while not stream.at_eof():
            data = await stream.readline()
            fp.write(data.decode())
            fp.flush()
        fp.close()

    def write_header(self):
        buffer = self.writer.write_header()
        self.proc.stdin.write(buffer)

    def write(self, timestamp: int, payload: bytes, media_type: FLVMediaType):
        buffer = self.writer.write(timestamp, payload, media_type)
        self.proc.stdin.write(buffer)

    async def close(self):
        await self.proc.stdin.drain()
        self.proc.stdin.close()
        await self.proc.wait()


class SimpleServer(SimpleRTMPServer):
    def __init__(self, output_directory: str):
        self.output_directory = output_directory
        super().__init__()

    async def create(self, host: str, port: int):
        loop = asyncio.get_event_loop()
        self.server = await loop.create_server(
            lambda: RTMPProtocol(controller=RTMP2SocketController(self.output_directory)),
            host=host,
            port=port,
        )


async def main():
    current_dir = os.path.dirname(os.path.abspath(__file__))
    server = SimpleServer(output_directory=current_dir)
    await server.create(host="0.0.0.0", port=args.port)
    await server.start()
    await server.wait_closed()


if __name__ == "__main__":
    asyncio.run(main())
